<?php

namespace App\Repositories;


use App\DTOs\AbstractDTO;
use App\Models\Book;
use App\Repositories\Interfaces\BookRepositoryInterface;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Pagination\LengthAwarePaginator;

class BookRepository extends Repository implements BookRepositoryInterface
{
    public function __construct(Book $model)
    {
        $this->setModel($model);
    }

    public function getIsFeatured(AbstractDTO $dto): Collection
    {
        return $this->getModel()->active()->isFeatured()
            ->when($dto->getLimit(), function ($query) use ($dto) {
                $query->limit($dto->getLimit());
            })
            ->orderBy('created_at', 'desc')
            ->get();
    }


    public function get(AbstractDTO $dto): Collection|LengthAwarePaginator
    {
        $books = $this->getModel()->active()->when($dto->getCategories(), function ($query) use ($dto) {
            $query->where('category_id', $dto->getCategories());
        })->orderByDesc('created_at')->newQuery();

        return $dto->getIsPaginated() ? $books->paginate($this->getPaginationLimit()) : $books->get();
    }

    public function getBySlug(string $slug): ?Model
    {
        return $this->getModel()->where('slug', $slug)->first();
    }

    public function getInterestedBooks(AbstractDTO $dto, string $categoryId): Collection
    {
        return $this->getModel()->active()->where('category_id', $categoryId)
            ->orderBy('created_at', 'desc')
            ->limit($dto->getLimit())
            ->get();
    }
}

