<?php

namespace App\Repositories;


use App\DTOs\AbstractDTO;
use App\Models\Blog;
use App\Repositories\Interfaces\BlogRepositoryInterface;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Pagination\LengthAwarePaginator;

class BlogRepository extends Repository implements BlogRepositoryInterface
{
    public function __construct(Blog $model)
    {
        $this->setModel($model);
    }

    /**
     * @param AbstractDTO $dto
     * @return Collection|LengthAwarePaginator
     */
    public function get(AbstractDTO $dto):  Collection|LengthAwarePaginator

    {
        $blogs= $this->getModel()->active()->isFeatured()
            ->when($dto->getLimit(), function ($query) use ($dto) {
                $query->limit($dto->getLimit());
            })
            ->when($dto->getCategories(), function ($query) use ($dto) {
                $query->where('category_id', $dto->getCategories());
            })
            ->when($dto->getSearch(), function ($query) use ($dto) {
                $query->where(function ($query) use ($dto) {
                    $searchTerm = $dto->getSearch();
                    $query->where('title_ar', 'LIKE', "%{$searchTerm}%")
                        ->orWhere('title_en', 'LIKE', "%{$searchTerm}%");
                });
            })->newquery();

        return $dto->getIsPaginated() ? $blogs->paginate($this->getPaginationLimit()) : $blogs->get();
    }


    public function getBySlug(string $slug): ?Model
    {
        return $this->getModel()->where('slug', $slug)->first();
    }

    public function getInterestedBlogs(AbstractDTO $dto, string $categoryId): Collection
    {
        return $this->getModel()->active()->where('category_id', $categoryId)
            ->orderBy('created_at', 'desc')
            ->limit($dto->getLimit())
            ->get();
    }
}

