<?php

namespace App\Http\Controllers\Website;

use App\DTOs\Api\Profile\GetProfileDto;
use App\DTOs\Api\Profile\UpdateProfileDto;
use App\DTOs\Api\Profile\UpdateProfilePasswordDto;
use App\Enums\Order\OrderStatus;
use App\Http\Controllers\Controller;
use App\Http\Requests\Website\UpdateAccountRequest;
use App\Http\Requests\Website\UpdatePasswordRequest;
use App\Services\General\GetAuthUserService;
use App\Services\Website\Profile\GetUserProfileService;
use App\Services\Website\Profile\UpdateUserPasswordService;
use App\Services\Website\Profile\UpdateUserProfileService;
use Exception;
use Illuminate\Contracts\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;

class ProfileController extends Controller
{
    public function __construct(
        private readonly GetUserProfileService     $getUserProfileService,
        private readonly UpdateUserProfileService  $updateUserProfileService,
        private readonly UpdateUserPasswordService $updateUserPasswordService,
        private readonly GetAuthUserService        $getAuthUserService,
    )
    {
    }

    /**
     * @throws Exception
     */
    public function orders(Request $request): View
    {
        $dto = new GetProfileDto($request->all());
        $user = $this->getAuthUserService->execute();
        $orders = $user->orders()->where('status', OrderStatus::PAID)->where('is_suspended', false)->when($request->search != "all", function ($query) {
            $query->where('orders.productable_type', 'App\Models\\' . \request('search'));
        })->get();

        return view('website.profile.profile-orders', [
            'orders' => $orders,
        ]);
    }


    /**
     * @throws Exception
     */
    public function index(Request $request): View
    {
        $user = $this->getAuthUserService->execute();


        return view('website.profile.my-profile', [
            'user' => $user,
        ]);
    }

    /**
     * @throws Exception
     */
    public function edit(Request $request): View
    {
        $user = $this->getAuthUserService->execute();

        return view('website.profile.edit', [
            'user' => $user,
        ]);
    }

    /**
     * @throws Exception
     */
    public function update(UpdateAccountRequest $request): RedirectResponse
    {
        try {
            $dto = new UpdateProfileDto($request->all());

            $user = $this->getAuthUserService->execute();

            $this->updateUserProfileService->setUser($user)->setDto($dto)->execute();
            toast(__('lang.account_updated_successfully'), 'success')->autoClose(9000);

            return redirect()->back();
        } catch (Exception $exception) {
            toast(__('something went wrong'), 'danger')->autoClose(9000);

            return redirect()->back();

        }

    }

    public function editPassword(): View
    {
        return view('website.profile.edit-password');
    }

    /**
     * @throws Exception
     */
    public function updatePassword(UpdatePasswordRequest $request)
    {
        try {
            $dto = new UpdateProfilePasswordDto($request->all());
            $user = $this->getAuthUserService->execute();

            $this->updateUserPasswordService->setUser($user)->setDto($dto)->execute();

            toast(__('lang.account_updated_successfully'), 'success')->autoClose(9000);

            return redirect()->back();

        } catch (Exception $exception) {
            toast($exception->getMessage(), 'error')->autoClose(9000);
            return redirect()->back();

        }
    }

}
