<?php

namespace App\Filament\Resources;

use App\Enums\Category\CategoryTypes;
use App\Enums\Media\MediaCollection;
use App\Filament\Resources\EventResource\Pages;
use App\Filament\Resources\EventResource\RelationManagers;
use App\Models\Event;
use Filament\Forms;
use Filament\Forms\Components\SpatieMediaLibraryFileUpload;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;

class EventResource extends Resource
{
    protected static ?string $model = Event::class;

    protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';
    protected static ?string $navigationGroup = 'Products';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Section::make()->schema([

                    Forms\Components\Select::make('category_id')
                        ->relationship('category', 'name_en', function ($query){
                            $query->where('type', CategoryTypes::EVENT->value);
                        })
                        ->label('Category')
                        ->required(),
                ]),

                Forms\Components\Section::make()->schema([

                    Forms\Components\TextInput::make('name_ar')
                        ->label('Name (Arabic)')
                        ->required()
                        ->maxLength(255)
                        ->live(onBlur: true)
                        ->afterStateUpdated(
                            fn(string $operation, $state, Forms\Set $set) => $set(
                                'slug',
                                slugArabic(title: $state)
                            )
                        ),

                    Forms\Components\TextInput::make('name_en')
                        ->label('Name (English)')
                        ->required()
                        ->maxLength(255),

                    Forms\Components\TextInput::make('slug')
                        ->label('Slug')
                        ->disabled()
                        ->dehydrated()
                        ->required()
                        ->unique(Event::class, 'slug', ignoreRecord: true),

                    Forms\Components\TextInput::make('price')
                        ->label('Price')
                        ->required()
                        ->numeric()
                        ->prefix('$'),

                    Forms\Components\TextInput::make('sale_price')
                        ->label('Sale Price')
                        ->columnSpanFull()
                        ->numeric()
                        ->prefix('$'),

                    Forms\Components\RichEditor::make('description_ar')
                        ->label('Description (Arabic)')
                        ->required()
                        ->maxLength(65535)
                        ->columnSpanFull(),
                    Forms\Components\RichEditor::make('description_en')
                        ->label('Description (English)')
                        ->required()
                        ->maxLength(65535)
                        ->columnSpanFull(),

                    SpatieMediaLibraryFileUpload::make('image')
                        ->label('Event Image')
                        ->collection(MediaCollection::EVENT_IMAGE->value)
                        ->columnSpanFull(),
                ]),

                Forms\Components\Section::make('Section One')->schema([

                    Forms\Components\Textarea::make('section_one_title_en')
                    ->label('Section One Title (English)')
                    ->required()
                    ->maxLength(65535)
                    ->columnSpanFull(),
                Forms\Components\Textarea::make('section_one_title_ar')
                    ->label('Section One Title (Arabic)')
                    ->required()
                    ->maxLength(65535)
                    ->columnSpanFull(),
                Forms\Components\RichEditor::make('section_one_description_en')
                    ->label('Section One Description (English)')
                    ->required()
                    ->maxLength(65535)
                    ->columnSpanFull(),
                Forms\Components\RichEditor::make('section_one_description_ar')
                    ->label('Section One Description (Arabic)')
                    ->required()
                    ->maxLength(65535)
                    ->columnSpanFull(),

                SpatieMediaLibraryFileUpload::make('section_one_image')
                    ->label('Section One Image')
                    ->name('section_one_image')
                    ->collection(MediaCollection::EVENT_SECTION_ONE_IMAGE->value)
                    ->columnSpanFull(),
                ]),

                Forms\Components\Section::make('Section Two')->schema([

                    Forms\Components\Textarea::make('section_two_title_en')
                    ->label('Section Two Title (English)')
                    ->required()
                    ->maxLength(65535)
                    ->columnSpanFull(),
                Forms\Components\Textarea::make('section_two_title_ar')
                    ->label('Section Two Title (Arabic)')
                    ->required()
                    ->maxLength(65535)
                    ->columnSpanFull(),
                Forms\Components\RichEditor::make('section_two_description_en')
                    ->label('Section Two Description (English)')
                    ->required()
                    ->columnSpanFull(),
                Forms\Components\RichEditor::make('section_two_description_ar')
                    ->label('Section Two Description (Arabic)')
                    ->required()
                    ->columnSpanFull(),

                SpatieMediaLibraryFileUpload::make('section_two_image')
                    ->label('Section Two Image')
                    ->name('section_two_image')
                    ->collection(MediaCollection::EVENT_SECTION_TWO_IMAGE->value)
                    ->columnSpanFull(),
                ]),

                Forms\Components\Section::make('Section Two')->schema([

                    Forms\Components\Textarea::make('section_three_title_en')
                    ->label('Section Three Title (English)')
                    ->required()
                    ->maxLength(65535)
                    ->columnSpanFull(),
                Forms\Components\Textarea::make('section_three_title_ar')
                    ->label('Section Three Title (Arabic)')
                    ->required()
                    ->maxLength(65535)
                    ->columnSpanFull(),
                Forms\Components\RichEditor::make('section_three_description_en')
                    ->label('Section Three Description (English)')
                    ->required()
                    ->maxLength(65535)
                    ->columnSpanFull(),
                Forms\Components\RichEditor::make('section_three_description_ar')
                    ->label('Section Three Description (Arabic)')
                    ->required()
                    ->maxLength(65535)
                    ->columnSpanFull(),

                SpatieMediaLibraryFileUpload::make('section_three_image')
                    ->label('Section Three Image')
                    ->name('section_three_image')
                    ->collection(MediaCollection::EVENT_SECTION_THREE_IMAGE->value)
                    ->columnSpanFull(),
                ]),
                Forms\Components\Section::make()->schema([

                    Forms\Components\Toggle::make('is_featured')
                        ->label('Is Featured')
                        ->required(),
                    Forms\Components\Toggle::make('status')
                        ->label('Status')
                        ->required(),
                ]),
            ]);

    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('name_ar')
                    ->label('Arabic Name')
                    ->searchable(),
                Tables\Columns\TextColumn::make('name_en')
                    ->label('English Name')
                    ->searchable(),
                Tables\Columns\TextColumn::make('category.name_en')
                    ->label('English Category')
                    ->searchable(),
                Tables\Columns\TextColumn::make('price')
                    ->money()
                    ->sortable(),
                Tables\Columns\ToggleColumn::make('status')
                    ->onColor('success')
                    ->offColor('danger')
                    ->searchable(),
                Tables\Columns\TextColumn::make('created_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('updated_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([

            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListEvents::route('/'),
            'create' => Pages\CreateEvent::route('/create'),
            'edit' => Pages\EditEvent::route('/{record}/edit'),
        ];
    }

    public static function canDelete(Model $record): bool
    {
        return false;
    }
}
