<?php

namespace App\Filament\Resources;

use App\Enums\Category\CategoryTypes;
use App\Enums\Media\MediaCollection;
use App\Filament\Resources\CourseResource\Pages;
use App\Filament\Resources\CourseResource\RelationManagers;
use App\Models\Course;
use Filament\Forms;
use Filament\Forms\Components\SpatieMediaLibraryFileUpload;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Model;

class CourseResource extends Resource
{
    protected static ?string $model = Course::class;

    protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';
    protected static ?string $navigationGroup = 'Products';

    public static function form(Form $form): Form
    {

        return $form
            ->schema([
                Forms\Components\Section::make()->schema([
                    Forms\Components\Select::make('category_id')
                        ->relationship('category', 'name_en', function ($query) {
                            $query->where('type', CategoryTypes::COURSE->value);
                        })
                        ->label('Category')
                        ->required(),
                ]),
                Forms\Components\Section::make()->columns(2)->schema([
                    Forms\Components\TextInput::make('name_ar')
                        ->label('Name (Arabic)')
                        ->required()
                        ->maxLength(255)
                        ->live(onBlur: true)
                        ->afterStateUpdated(
                            fn(string $operation, $state, Forms\Set $set) => $set(
                                'slug',
                                slugArabic(title: $state)
                            )
                        ),

                    Forms\Components\TextInput::make('name_en')
                        ->label('Name (English)')
                        ->required()
                        ->maxLength(255),

                    Forms\Components\TextInput::make('slug')
                        ->label('Slug')
                        ->disabled()
                        ->dehydrated()
                        ->required()
                        ->unique(Course::class, 'slug', ignoreRecord: true),
                ]),
                Forms\Components\Section::make()->schema([
                    Forms\Components\RichEditor::make('short_description_ar')
                        ->label('Short Description (Arabic)')
                        ->required()
                        ->maxLength(65535)
                        ->columnSpanFull(),
                    Forms\Components\RichEditor::make('short_description_en')
                        ->label('Short Description (English)')
                        ->required()
                        ->maxLength(65535)
                        ->columnSpanFull(),
                    Forms\Components\RichEditor::make('long_description_ar')
                        ->label('Long Description (Arabic)')
                        ->required()
                        ->columnSpanFull(),
                    Forms\Components\RichEditor::make('long_description_en')
                        ->label('Long Description (English)')
                        ->required()
                        ->columnSpanFull(),
                ]),
                Forms\Components\Section::make()->schema([
                    Forms\Components\TextInput::make('price')
                        ->label('Price')
                        ->required()
                        ->columnSpanFull()
                        ->numeric()
                        ->prefix('$'),

                    Forms\Components\TextInput::make('sale_price')
                        ->label('Sale Price')
                        ->columnSpanFull()
                        ->numeric()
                        ->prefix('$'),
                ]),
                Forms\Components\Section::make()->schema([
                    Forms\Components\TextInput::make('first_highlight_ar')
                        ->label('First Course Description Highlight Name (Arabic)')
                        ->required()
                        ->maxLength(255),

                    Forms\Components\TextInput::make('first_highlight_en')
                        ->label('First Course Description Highlight Name (English)')
                        ->required()
                        ->maxLength(255),

                    Forms\Components\RichEditor::make('objectives_ar')
                        ->label('First Course Description Highlight Content (Arabic)')
                        ->required()
                        ->columnSpanFull(),
                    Forms\Components\RichEditor::make('objectives_en')
                        ->label('First Course Description Highlight Content (English)')
                        ->required()
                        ->columnSpanFull(),
                ]),

                Forms\Components\Section::make()->schema([
                    Forms\Components\TextInput::make('second_highlight_ar')
                        ->label('Second Course Description Highlight Name (Arabic)')
                        ->required()
                        ->maxLength(255),

                    Forms\Components\TextInput::make('second_highlight_en')
                        ->label('Second Course Description Highlight Name (English)')
                        ->required()
                        ->maxLength(255),

                    Forms\Components\RichEditor::make('course_highlights_ar')
                        ->label('Second Course Description Highlight Content (Arabic)')
                        ->required()
                        ->columnSpanFull(),
                    Forms\Components\RichEditor::make('course_highlights_en')
                        ->label('Second Course Description Highlight Content (English)')
                        ->required()
                        ->columnSpanFull(),
                ]),
                Forms\Components\Section::make()->columns(2)->schema([
                    Forms\Components\TextInput::make('earn_ar')
                        ->label('Earnings (Arabic)')
                        ->required()
                        ->maxLength(255),
                    Forms\Components\TextInput::make('earn_en')
                        ->label('Earnings (English)')
                        ->required()
                        ->maxLength(255),
                    Forms\Components\TextInput::make('time')
                        ->label('Time')
                        ->required()
                        ->maxLength(255),
                    Forms\Components\TextInput::make('age_from')
                        ->label('Age From')
                        ->required()
                        ->numeric(),
                    Forms\Components\TextInput::make('age_to')
                        ->label('Age To')
                        ->required()
                        ->numeric(),
                    Forms\Components\TextInput::make('prerequisites_ar')
                        ->label('Prerequisites (Arabic)')
                        ->required()
                        ->maxLength(255),
                    Forms\Components\TextInput::make('prerequisites_en')
                        ->label('Prerequisites (English)')
                        ->required()
                        ->maxLength(255),
                ]),
                Forms\Components\Section::make()->columns(2)->schema([
                    Forms\Components\TextInput::make('instructor_name_ar')
                        ->label('Instructor Name (Arabic)')
                        ->required()
                        ->maxLength(255),
                    Forms\Components\TextInput::make('instructor_name_en')
                        ->label('Instructor Name (English)')
                        ->required()
                        ->maxLength(255),
                    Forms\Components\TextInput::make('instructor_position_ar')
                        ->label('Instructor Position (Arabic)')
                        ->required()
                        ->maxLength(255),
                    Forms\Components\TextInput::make('instructor_position_en')
                        ->label('Instructor Position (English)')
                        ->required()
                        ->maxLength(255),
                    Forms\Components\RichEditor::make('instructor_short_description_ar')
                        ->label('Instructor Short Description (Arabic)')
                        ->required()
                        ->maxLength(255),
                    Forms\Components\RichEditor::make('instructor_short_description_en')
                        ->label('Instructor Short Description (English)')
                        ->required()
                        ->maxLength(255),
                    Forms\Components\RichEditor::make('instructor_long_description_ar')
                        ->label('Instructor Long Description (Arabic)')
                        ->required()
                        ->maxLength(255),
                    Forms\Components\RichEditor::make('instructor_long_description_en')
                        ->label('Instructor Long Description (English)')
                        ->required()
                        ->maxLength(255),
                    SpatieMediaLibraryFileUpload::make('instructor_image')
                        ->label('Instructor Image')
                        ->collection(MediaCollection::COURSE_INSTRUCTOR_IMAGE->value)
                        ->columnSpanFull(),
                ]),
                Forms\Components\Section::make()->schema([
                    Forms\Components\Toggle::make('is_new')
                        ->label('Is New')
                        ->required(),
                    Forms\Components\Toggle::make('is_featured')
                        ->label('Is Featured')
                        ->required(),
                    Forms\Components\Toggle::make('status')
                        ->label('Status')
                        ->required(),
                ]),

                Forms\Components\Section::make("Home Page Images")->schema([
                    SpatieMediaLibraryFileUpload::make('course_home_english_image')
                        ->label('English Image')
                        ->collection(MediaCollection::HOME_COURSE_IMAGE_ENGLISH->value)
                        ->columnSpanFull(),
                    SpatieMediaLibraryFileUpload::make('course_home_arabic_image')
                        ->label('Arabic Image')
                        ->collection(MediaCollection::HOME_COURSE_IMAGE_ARABIC->value)
                        ->columnSpanFull(),
                ]),

                Forms\Components\Section::make("List Courses Page Images")->schema([
                    SpatieMediaLibraryFileUpload::make('course_list_english_image')
                        ->label('English Image')
                        ->collection(MediaCollection::LIST_COURSE_IMAGE_ENGLISH->value)
                        ->columnSpanFull(),
                    SpatieMediaLibraryFileUpload::make('course_list_arabic_image')
                        ->label('Arabic Image')
                        ->collection(MediaCollection::LIST_COURSE_IMAGE_ARABIC->value)
                        ->columnSpanFull(),
                ]),

                Forms\Components\Section::make("Course Details Page Images")->schema([
                    SpatieMediaLibraryFileUpload::make('course_inner_english_image')
                        ->label('English Image')
                        ->collection(MediaCollection::INNER_COURSE_IMAGE_ENGLISH->value)
                        ->columnSpanFull(),
                    SpatieMediaLibraryFileUpload::make('course_inner_arabic_image')
                        ->label('Arabic Image')
                        ->collection(MediaCollection::INNER_COURSE_IMAGE_ARABIC->value)
                        ->columnSpanFull(),
                ]),
            ]);

    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('name_ar')
                    ->label('Arabic Name')
                    ->searchable(),
                Tables\Columns\TextColumn::make('name_en')
                    ->label('English Name')
                    ->searchable(),
                Tables\Columns\TextColumn::make('category.name_en')
                    ->label('English Category')
                    ->searchable(),
                Tables\Columns\TextColumn::make('price')
                    ->label('Price')
                    ->money()
                    ->sortable(),
                Tables\Columns\TextColumn::make('sale_price')
                    ->label('Sale Price')
                    ->money()
                    ->sortable(),
                Tables\Columns\ToggleColumn::make('is_new')
                    ->onColor('success')
                    ->offColor('danger')
                    ->searchable(),
                Tables\Columns\ToggleColumn::make('is_featured')
                    ->onColor('success')
                    ->offColor('danger')
                    ->searchable(),
                Tables\Columns\ToggleColumn::make('status')
                    ->onColor('success')
                    ->offColor('danger')
                    ->searchable(),
                Tables\Columns\TextColumn::make('created_at')
                    ->label('Created At')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('updated_at')
                    ->label('Updated At')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                // Add filters if needed
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([

            ]);
    }

    public static function getRelations(): array
    {
        return [
            RelationManagers\VideosRelationManager::class
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListCourses::route('/'),
            'create' => Pages\CreateCourse::route('/create'),
            'edit' => Pages\EditCourse::route('/{record}/edit'),
        ];
    }

    public static function canDelete(Model $record): bool
    {
        return false;
    }
}
