<?php

namespace App\Filament\Resources;

use AmidEsfahani\FilamentTinyEditor\TinyEditor;
use App\Enums\Category\CategoryTypes;
use App\Enums\Media\MediaCollection;
use App\Filament\Resources\BlogResource\Pages;
use App\Filament\Resources\BlogResource\RelationManagers;
use App\Models\Blog;
use Filament\Forms;
use Filament\Forms\Components\SpatieMediaLibraryFileUpload;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use RalphJSmit\Filament\SEO\SEO;

class BlogResource extends Resource
{
    protected static ?string $model = Blog::class;

    protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';
    protected static ?string $navigationGroup = 'Home';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Select::make('category_id')
                    ->relationship('category', 'name_en', function ($query){
                        $query->where('type', CategoryTypes::BLOG->value);
                    })
                    ->required(),

                Forms\Components\Section::make()->columns(2)->schema([

                    Forms\Components\TextInput::make('title_ar')
                        ->required()
                        ->maxLength(255)
                        ->afterStateUpdated(
                            fn(string $operation, $state, Forms\Set $set) => $set(
                                'slug',
                                slugArabic(title: $state)
                            )
                        ),

                    Forms\Components\TextInput::make('title_en')
                        ->required()
                        ->maxLength(255),

                    Forms\Components\TextInput::make('slug')
                        ->disabled()
                        ->dehydrated()
                        ->required()
                        ->unique(Blog::class, 'slug', ignoreRecord: true),


                ]),

                Forms\Components\Section::make('Blog Seo')->columns(2)->schema([
                    SEO::make(),
                ]),

                Forms\Components\Section::make()->columns(2)->schema([

                    Forms\Components\TextInput::make('author_ar')
                        ->required()
                        ->maxLength(255),
                    Forms\Components\TextInput::make('author_en')
                        ->required()
                        ->maxLength(255),
                ]),

                Forms\Components\Section::make()->columns(2)->schema([


                    TinyEditor::make('content_ar')
                        ->profile('default|simple|full|minimal|none|custom')
                        ->direction('auto') // Set RTL or use ->direction('auto|rtl|ltr')
                        ->columnSpan('full')
                        ->required(),

                    TinyEditor::make('content_en')
                        ->profile('default|simple|full|minimal|none|custom')
                        ->direction('auto') // Set RTL or use ->direction('auto|rtl|ltr')
                        ->columnSpan('full')
                        ->required(),

                ]),


                Forms\Components\Section::make()->columns(2)->schema([
                    Forms\Components\Toggle::make('status')
                        ->label('Status')
                        ->required(),

                    Forms\Components\Toggle::make('is_featured')
                        ->label('Is Featured')
                        ->required(),
                ]),

                Forms\Components\Section::make("Images Section")->columns(2)->schema([

                    SpatieMediaLibraryFileUpload::make('english_image')
                        ->label('English Image')
                        ->collection(MediaCollection::BLOG_IMAGE_ENGLISH->value)
                        ->columnSpanFull(),

                    SpatieMediaLibraryFileUpload::make('arabic_image')
                        ->label('Arabic Image')
                        ->collection(MediaCollection::BLOG_IMAGE_ARABIC->value)
                        ->columnSpanFull(),
                ]),

                Forms\Components\Section::make("List Blog Page Images")->schema([
                    SpatieMediaLibraryFileUpload::make('blog_list_english_image')
                        ->label('English Image')
                        ->collection(MediaCollection::LIST_BLOG_IMAGE_ENGLISH->value)
                        ->columnSpanFull(),
                    SpatieMediaLibraryFileUpload::make('blog_list_arabic_image')
                        ->label('Arabic Image')
                        ->collection(MediaCollection::LIST_BLOG_IMAGE_ARABIC->value)
                        ->columnSpanFull(),
                ]),

                Forms\Components\Section::make("Blog Details Page Images")->schema([
                    SpatieMediaLibraryFileUpload::make('blog_inner_english_image')
                        ->label('English Image')
                        ->collection(MediaCollection::INNER_BLOG_IMAGE_ENGLISH->value)
                        ->columnSpanFull(),
                    SpatieMediaLibraryFileUpload::make('blog_inner_arabic_image')
                        ->label('Arabic Image')
                        ->collection(MediaCollection::INNER_BLOG_IMAGE_ARABIC->value)
                        ->columnSpanFull(),
                ]),

                Forms\Components\Section::make("Video Section")->columns(2)->schema([

                    Forms\Components\Select::make('video_type')
                        ->options([
                            'youtube' => 'youtube',
                            'vimeo' => "vimeo"
                        ])
                        ->required(),

                    Forms\Components\TextInput::make('video_url')
                        ->required()
                        ->maxLength(65535)
                        ->columnSpanFull(),

                    SpatieMediaLibraryFileUpload::make('video_image')
                        ->label('video image')
                        ->collection(MediaCollection::BLOG_VIDEO_IMAGE->value)
                        ->columnSpanFull(),
                ]),

            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('category.name_en')
                    ->searchable(),
                Tables\Columns\TextColumn::make('title_ar')
                    ->searchable(),
                Tables\Columns\TextColumn::make('title_en')
                    ->searchable(),
                Tables\Columns\TextColumn::make('author_ar')
                    ->searchable(),
                Tables\Columns\TextColumn::make('author_en')
                    ->searchable(),

                Tables\Columns\ToggleColumn::make('status')
                    ->onColor('success')
                    ->offColor('danger')
                    ->searchable(),

                Tables\Columns\TextColumn::make('created_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('updated_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListBlogs::route('/'),
            'create' => Pages\CreateBlog::route('/create'),
            'edit' => Pages\EditBlog::route('/{record}/edit'),
        ];
    }
}
